#import "@preview/polylux:0.3.1": *
#import themes.metropolis: *

#let title = "Practical lattice reductions"
#let subtitle = "for CTF challenges"
#set par(justify: true)
#set text(size: 24pt)
#set list(marker: level => (sym.bullet, sym.compose, sym.star.filled).at(calc.rem(level, 3)))

#let imgorigin(url) = text(fill: gray.lighten(30%), size: 12pt, link(url))
#let vec(x) = $bold(#x)$
#let vB = vec[B]
#let va = vec[a]
#let vb = vec[b]
#let vt = vec[t]
#let vv = vec[v]
#let lat = math.cal[L]
#let pub(x) = text(fill: blue, x)
#let priv(x) = text(fill: red, x)
#show: metropolis-theme.with(aspect-ratio: "4-3", footer: [#title #subtitle#h(1fr)Athens, 2024/02/23])

#title-slide(title: title, subtitle: subtitle, author: "Robin Jadoul", date: "2024/02/23", extra: "Bootcamp ICC Team Europe, Athens, Greece")

#slide(title: [Outline])[
= Outline
#v(2em)
#metropolis-outline
]

#new-section-slide[Lattices?]
#slide(title: [Lattices])[
  - "A lattice $lat$ of dimension $n$ is a discrete additive subgroup of $RR^n$."
  - It's a group $=>$ addition, scalar mult
  - Discrete $=>$ we can map it to $QQ^n$ or $ZZ^n$
  - A group is a $ZZ$-module, so think vector spaces
  - Pick a basis $vB = (vb_1, ..., vb_n) in RR^n$
    - #alert[We usually write $vb_i$ as rows of $vB$]
  - $lat = {sum a_i dot vb_i bar va in ZZ^n}$
  - Many choices of $vB$
]
#slide(title: [Lattices])[
  #set align(center + horizon)
  #side-by-side[
    #image("lattice_irl.jpg")#imgorigin("https://www.redwoodstore.com/spectre-fan-lattice")
  ][
    #image("lattice_2d_plain.svg")#imgorigin("https://en.wikipedia.org/wiki/Lattice_%28group%29")
  ][
    #image("lattice_3d.png")#imgorigin("https://en.wikipedia.org/wiki/File:Diamond_lattice.stl")
  ]
]
#slide(title: [Lattices])[
  #set align(center + horizon)
  #image("lattice_2d_bases.svg", width: 70%)#imgorigin("https://en.wikipedia.org/wiki/Lattice_reduction")
]

#slide(title: [Lattice properties])[
  - "Fundamental parallelepiped" $cal(P)(vB)$: a single "enclosed region"
    - $cal(P)(vB) = {sum a_i dot vb_i bar va in [0, 1)}$
    - $RR^n$ is tiled by $cal(P)(vB)$
  - $det(lat) = upright("vol")(cal(P)(vB))) = abs(det(vB))$
    - Invariant, independent of $vB$
    - Base change: invertible and unimodular
  - Successive minima: $lambda_(i)(lat)$
    - $lambda_1(lat) $ length of #alert[shortest vector]
    - $lambda_1(lat) <= sqrt(n) abs(det(lat))^(1/n)$
    - $"GM"(lambda_1, ..., lambda_n) = (product lambda_i)^(1/n) <= sqrt(n) abs(det(lat))^(1/n)$
  - Distance $mu(vt, lat) = display(min_(vv in lat)) norm(vt - vv)$
]

#slide(title: [Lattice properties])[
  #set align(center + horizon)
  #side-by-side[
    #image("fundamental_region.png", width: 90%)#imgorigin("https://simons.berkeley.edu/sites/default/files/docs/14953/intro.pdf")
  ][
    #image("successive_minima.png", width: 90%)#imgorigin("https://simons.berkeley.edu/sites/default/files/docs/14953/intro.pdf")
  ]
]

#slide(title: [Sage])[
  ```py
  from sage.modules.free_module_integer import IntegerLattice
  B = Matrix(QQ, [[1, 0], [0, 2]])/2
  L = IntegerLattice(B.denominator() * B, lll_reduce=False)
  # Warning, may be slow :)
  L.shortest_vector() # (1, 0)
  L.closest_vector((123/42, 345/12)) # (3, 28)

  L.volume() # 2
  ```
]

#new-section-slide[Why lattices?]
#slide(title: [Why lattices? Part 1: construction])[
  - Many things in lattices are hard
  - SVP: "shortest vector problem"
  - CVP: "closest vector problem"
  - SIS: "short integer solutions"
  - $=>$ build trapdoors, e.g. LWE
  - Hopefully post-quantum too
  - See later
]

#slide(title: [Why lattices? Part 2: destruction])[
  - Many things are "small"
  - Many things are discrete
  - e.g. some instances of integer programming
  - RSA: $p q - phi(p q) = p + q - 1$ is "small" wrt. $p q$
  - Breaking lattice schemes
  - Generally: linear structure
  - #alert[Think about *linear systems* with small solutions]
]
#slide(title: [Why lattices? Part 2: destruction])[
  - Known to break many crypto "weaknesses"
  - Some bias in your RNG? Lattices will break it.
  - Chose your RSA private key wrong? Lattices will break it.
  - Lost some precision in your floating points calculations? Lattices might help.
  - Some think they even might break factoring :)
]

#new-section-slide[How lattices?]

#slide(title: [Improving a basis])[
  - Starting point: _some_ basis $vB$
  - Goal: _good_ basis $vB'$
  - But what is good?
  - And how do we find it?
]

#slide(title: [Good lattices])[
  - Goal: find a better basis
  - Good basis?
    - Shorter basis vectors
    - Close to orthogonal
    - Find some short vectors
  - Great basis?
    - Read off $lambda_1(lat)$
    - Read off all $lambda_(i)(lat)$?
]

#slide(title: [Intuition from linear algebra])[
  - We know $det(lat)$ is constant
  - Want: shorter $vb_i$
  - So we need wider angles between all $vb_i$ to have more area
  - Hence, more orthogonal
  #v(1em)
  - Gram-Schmidt orthogonalization
  - But breaks the lattice
  - Use it as a guideline
]

#slide(title: [Lattice reduction algorithms])[
  - LLL (Lenstra–Lenstra–Lovász)
    - Polynomial time $cal(O)(n^6 log^3 norm(vB)_oo)$
    - $norm(vb'_1) <= 2^((n - 1)/2) lambda_1(lat)$
  - HKZ (Hermite–Korkine–Zolotarev)
    - Exponential time
    - $norm(vb'_1) = lambda_1(lat)$
  - BKZ (Block (H)KZ)
    - Parametrized by block size $beta$
    - Larger $beta$: slower
    - Smaller $beta$: worse basis
  - Sieving and other costly approaches
]

#slide(title: [Basis reduction in 2D])[
  - In two dimensions, exact is easy
  - Provides some basic intuition for LLL
  - Looks like GCD

  ```py
  def gauss_reduction(v1, v2):
    while True:
        if v2.norm() < v1.norm():
            v1, v2 = v2, v1 # swap step
        m = round( (v1 * v2) / (v1 * v1) )
        if m == 0:
            return (v1, v2)
        v2 = v2 - m*v1 # reduction step
  ```
]

#slide(title: [A brief look at LLL])[
  #set text(size: 12pt)
  ```py
def LLL(B, delta):
    Q = gram_schmidt(B)

    def mu(i,j):
        v = B[i]
        u = Q[j]
        return (v*u) / (u*u)

    n, k = B.nrows(), 1
    while k < n:

        # length reduction step
        for j in reversed(range(k)):
            if abs(mu(k,j)) > .5:
                B[k] = B[k] - round(mu(k,j))*B[j]
                Q = gram_schmidt(B)

        # swap step
        if Q[k]*Q[k] >= (delta - mu(k,k-1)**2)*(Q[k-1]*Q[k-1]):
            k = k + 1
        else:
            B[k], B[k-1] = B[k-1], B[k]
            Q = gram_schmidt(B)
            k = max(k-1, 1)

   return B

  ```
]


#new-section-slide[Lattice tips and tricks]
#slide(title: [Weights])[
  $
    vec(z) = a_1 vv_1 + ... + a_m vv_m
  $
  - $m > n$
  - All $a_i$ small
  - Linear system with a small solution
    - But underdetermined
    - So not just linear algebra
  - (approximate) SVP would find a short solution
]
#slide(title: [Weights])[
  $
    mat(
      vec(z), 0, 0, ..., 0;
      -vv_1, 1, 0, ..., 0;
      -vv_2, 0, 1, ..., 0;
      dots.v, dots.v, dots.v, dots.down, dots.v;
      -vv_m, 0, 0, ..., 1;
    )
  $
  - *But*, what if we have some remaining _short_ $vec(r) = vec(z) - sum a_i vv_i$?
  - To LLL, short is short
  - Assign higher weights $W$ to first columns
    - $=> W vec(r)$ not so short anymore
    - Could even vary $W_i$ for element of $vec(z)$
  - #alert[Note: ] short vectors can also be the negation of what you search
]
#slide(title: [Weights])[
  $
    mat(
      W vec(z), 0, 0, ..., 0;
      -W vv_1, 1, 0, ..., 0;
      -W vv_2, 0, 1, ..., 0;
      dots.v, dots.v, dots.v, dots.down, dots.v;
      -W vv_m, 0, 0, ..., 1;
    )
  $
  #line(length: 100%)
  ```py
  z = vector(ZZ, [...])
  v = Matrix(ZZ, m, n, [[...], ...])
  L = Matrix.block([[Matrix(z), 0], [v, 1]])
  W = Matrix.diagonal([w1, w2, ..., wn, 1, 1, ..., 1])
  L = (L * W).LLL() / W
  ```
]
#slide(title: [How to determine weights])[
  + Wing it and hope for the best :)
    - Works fairly often
    - Can finetune with synthetic data
    - Wiggling weights can even help finding different solutions
  + Investigate expected weights
    - Mostly when different columns have different expected weights in the target vector
    - Observe: outliers in $norm(vv) = sqrt(v_1^2 + ... + v_n^2)$ weigh more
    - So the goal is: make all $v_i$ roughly equal
      - Investigate expected result in target vector
      - Modify weights per column so target vector is all $1$ (or arbitrary constant like $2^(128)$)
]

#slide(title: [CVP])[
  - Sometimes, switch view to CVP
  - Rather than solving a linear system, you're close to some lattice point
  - e.g. integer multiple + random noise (see LWE later)
  - So looking for a lattice vector close to our target
    - Either close vector is final goal
    - Or just solve with linear algebra after
  - If you have a range of values, put the target in the center
]
#slide(title: [How to CVP])[
  == Kannan embedding
  $
    mat(vB, 0; vt, q)
  $
  - Embed CVP into an SVP instance
  - Likely close to what you started with
  - Short vector: $(vt - vB vec(c), q)$
  - $q space ~ $ a weight, matters for results
]
#slide(title: [How to CVP])[
  == Babai's closest plane
  - Uses a reduced basis for the original lattice
  - Greedy algorithm
  - Iteratively project each coordinate onto the closest hyperplane
  - In sage, over $QQ$: GS step is generally slow
    - Exact numbers that grow fast-ish
  ```py
def Babai_CVP(mat, target):
    M = IntegerLattice(mat, lll_reduce=True).reduced_basis
    G = M.gram_schmidt()[0]
    diff = target
    for i in reversed(range(G.nrows())):
      	diff -=  M[i] * ((diff * G[i]) / (G[i] * G[i])).round()
    return target - diff
  ```
]

#slide(title: [Try different reductions])[
  - When LLL is fast enough, but gives no results
  - Consider trying BKZ instead
  - Experiment with block size $beta$, synthetic data is good
  - $(beta = n) equiv$ HKZ
  - For more speed (especially coppersmith): consider `flatter`
]

#slide(title: [Magic tricks for fast exploration])[
  - Got a linear system and some bounds?
  - Why not try asking nicely
  - `rkm0959` made a tool/library: #link("https://github.com/rkm0959/Inequality_Solving_with_CVP")[rkm0959/Inequality_Solving_with_CVP]
  - Could even make a wrapper for convenience
  - Good for first exploration, not always foolproof
]

#slide(title: [Enumeration])[
  - Your target is not always the shortest
  - Or even in the basis for that matter
  - It's still short though
    - It's a small linear combination of basis vectors
    - Try bruteforce
    - Or random combinations
  - fp(y)lll also has structured enumeration
  - Optionally with extra pruning
  - badly documented
    - https://fpylll.readthedocs.io/en/latest/modules.html
]
#slide(title: [fpylll enumeration])[
  ```py
from fpylll import IntegerMatrix
from fpylll.fplll.gso import MatGSO
from fpylll.fplll.enumeration import (Enumeration,
                                      EvaluatorStrategy)
A = IntegerMatrix.from_matrix(M.LLL())
count = 2000
G = MatGSO(A)
G.update_gso()
enum = Enumeration(G, nr_solutions=count,
          strategy=EvaluatorStrategy.BEST_N_SOLUTIONS)
n = M.nrows()
for vec, length in enum.enumerate(0, n, max_dist, 0, t):
  ...
  ```
]

#slide(title: [Polynomials])[
  - Polynomials form a vector space
    - If degree is bounded/fixed
    - Over $RR$ or otherwise
  - So we can take a discrete additive subgroup of them
  - Look, ma, it's a lattice!
  - Basis for coppersmith's method and ring-LWE (see later)
]

#slide(title: [Codes])[
  - Hmm, I have a lattice over $FF_2$
  - While that may be true, "small" mostly breaks down
  - Have a look at coding theory instead
  - Techniques like _ISD_ can be very powerful here
  - Can also apply over
    - $FF_3$ or other small fields
    - Fields of small characteristic ($FF_(2^k)$ etc)
]

#new-section-slide[Common lattice problems]
#slide(title: [Making things modular])[
  - Instead of working over $ZZ$, we now want $ZZ\/q ZZ$
  - Keep thinking about linear systems
  - $sum a_i x_i equiv y quad (mod q)$
  - $sum a_i x_i = y + k q$
  - Repeat a few times
  - Stack $q dot I_m$ under your matrix
]

#slide(title: [Knapsack and Subset Sum])[
  - Given:
    - A set $S = {s_1, ..., s_n}$
    - A value $v = sum b_i s_i$, with $b_i in {0, 1}$
  - Find appropriate $b_i$
  - Often called a knapsack problem
  - More accurately it's a subset sum problem
    - there are no values attached
  - Known public key cryptosystem
    - Merkle-Damgård
    - Broken by lattices (low density)
]
#slide(title: [Knapsack and Subset Sum])[
  $
    mat(
      v, 0, 0, ..., 0;
      -s_1, 1, 0, ..., 0;
      -s_2, 0, 1, ..., 0;
      dots.v, dots.v, dots.v, dots.down, dots.v;
      -s_n, 0, 0, ..., 1;
    )
  $
  - Short vector $(0, b_1, b_2, ..., b_n)$
  - Rephrase as CVP
    - Leave out first row
    - Target: $(v, 0, 0, ..., 0)$
  - Optimize CVP:
    - Want $b_i in {0, 1}$, so centered around $1/2$
    - Can do the same trick in the original lattice
]
#slide(title: [Knapsack and Subset Sum])[
=== Think about it:
- What about a more general version
  - $b_i in cal(X)$
  - Knapsack: optimize for some value $t_i$
  - Dealing with negative numbers
  - Parallel instances
  - Modular
  - ...
- _Hidden_ Subset Sum problem
#h(1em)
- Don't forget to look at the negatives in your reduced basis!
]

#slide(title: [Approximate GCD])[
  - Given: samples $x_i = q_i p + r_i$, with small $r_i$
  - Target: Find $p$, the gcd of the samples, up to errors $r_i$
  - _Partial_ AGCD: $r_0 = 0$
    - i.e. $x_0 = q_0 p$
    - e.g. RSA with extra information
]
#slide(title: [AGCD: SDA])[
  - $x_i/x_0 approx q_i/q_0$
  - Find candidate $q_0$
  - Recover $p$ from $x_0, q+0$
  - Short vector: $(W q_0, q_0 r_1 - q_1 r_0, ...)$
  $
    mat(
      W, x_1, x_2, ..., x_n;
      0, x_0, 0, ..., 0;
      0, 0, x_0, ..., 0;
      dots.v, dots.v, dots.v, dots.down, dots.v;
      0, 0, 0, ..., x_0;
    )
  $
]
#slide(title: [AGCD: Orthogonal lattice])[
  - Orthogonal lattice
    - $lat^perp = {vv | forall b in lat, lr(angle.l vv, vb angle.r) = 0}$
    - Observe: $lat subset.eq (lat^perp)^perp$
    - Useful for some problems, including hidden subset sum
  - General idea: find short vector _orthogonal_ to some target
    - Often just to one vector or a lattice with 2 basis vectors
    - Then derive some useful quantity
  - $lat(vec(q), vec(r))^perp subset.eq lat(vec(x))^perp$, and short
  - reduce $lat(vec(x))^perp$ to find a sub-basis spanning $lat(vec(q), vec(r))$
  - recover $vec(q), vec(r)$
]


#slide(title: [Hidden Number Problem])[
  $
    pub(alpha_i) priv(x) + pub(rho_i) priv(k_i) equiv pub(beta_i) quad (mod N)
  $
  - $k_i$ bounded
  - See (EC)DSA biased nonce attacks
    - $alpha_i = -r_i$, $rho_i = s_i$, $beta_i = H - 2^t "MSB"_"nonce"$, $k_i = "LSB"_"nonce"$
  - Key realization: $k_i equiv (beta_i - alpha_i x)/(rho_i)$ is bounded/small
    - Try to build the lattice ;)
    #[#set text(size: 14pt, fill: gray)
    - Or read biased nonce papers]
  - Generalization: EHNP
    - Support multiple "holes"
    - Formulation gets complex
    - "Just" implementing the paper is feasible
]

#slide(title: [Coppersmith's Method])[
  - Shift in focus
    - No longer linear systems... one polynomial
  - $f(x) equiv 0 quad (mod N)$, monic, $x < X$ bounded
    - Or even $mod d approx N^beta$ with $d | N$
  - Sage has `f.small_roots()`, with some parameters
    - or use implementation from `kiona`/`defund`/...
    - `flatter` usually works very well for these \
  - $X < N^(beta^2/deg(f) - epsilon)$
    - $epsilon$ is a useful parameter for sage
    - Smaller $epsilon$ is slower, maybe brute some bits
  - Multivariate (heuristic) generalizations
]
#slide(title: [Coppersmith intuition])[
  - Generate polynomials sharing roots $(mod N)$
    - $x^k f(x)$
    - $N^k f(x)$
    - $f^k (x)$
    - $=> x^i N^j f^k (x)$
  - Find small $f'$ over $ZZ$
    - Lattice reduction
    - Factor over $ZZ$
    - Check results $(mod N)$
  - Multivariate
    - Extract roots from multiple polynomials
    - Gröbner basis, resultants, ...
]
#slide(title: [Coppersmith attacks])[
  - RSA: stereotyped message
    - $f(x) = (K + x)^e - c equiv 0 quad (mod N)$, $x$ small
  - RSA: partially known factor
    - $f(x) = (p_"high" + x) equiv 0 quad (mod p)$, $x < N^(1/4)$, $p | N$
  - Boneh-Durfee
    - $f(x, y) = x ((N + 1) - y) equiv 0 quad (mod e)$
    - $y = -(p + q)$, $x$ modular "wraps"
  - AGCD
    - Multivariate
]


#new-section-slide[Building with lattices]
#slide(title: [Learning With Errors])[
  $
    priv(vec(s)) &<- chi_upright(k)^n\
    pub(vec(a_i)) &<- ZZ_q^n\
    e_i &<- chi_upright(e)\
    pub(b_i) &= lr(angle.l vec(s), va_i angle.r) + e_i
  $
  #line(length: 100%)
  $
    vb = pub(vec(A)) priv(s) + vec(e)
  $
]
#slide(title: [LWE])[
  - Distinguishing
  - Key recovery
  - Embedding messages
    - $b = lr(angle.l vec(s), va angle.r) + e + q/p dot m$
    - $b = lr(angle.l vec(s), va angle.r) + p dot e + m$
]

#slide(title: [Ring-LWE])[
  $
    R &= ZZ[X]\/f(X), f "monic irreducible", deg(f) = N\
    R_q &= R\/q R\
    s(X) &<- chi_upright(k)^N in R_q\
    e(X) &<- chi_upright(e)^N in R_q\
    b_i (X) &= a_i (X) dot s(X) +  e_i (X)
  $
]

#slide(title: [Ring-LWE])[
  $
    b_i (X) = a_i (X) dot s(X) + e_i (X)
  $
  #line(length: 100%)
  $
    vec(b_i) = mat(
      a_(i, 1), (X^(-1) a)_1, ..., (X^(-N + 1) a)_1;
      a_(i, 2), (X^(-1) a)_2, ..., (X^(-N + 1) a)_2;
      dots.v, dots.v, dots.down, dots.v;
      a_(i, N), (X^(-1) a)_N, ..., (X^(-N + 1) a)_N;
    ) dot vec(s) + vec(e_i)
  $
]

#slide(title: [NTRU])[
  $
    R &= ZZ[X]\/(X^N plus.minus 1)\
    R_q &= R\/q R\
    priv(f) &<- chi^N in R_q, exists f^(-1)\
    g &<- chi^N in R_q\
    pub(h) &= g/f
  $
]
#slide(title: [NTRU])[
  - Distinguishing
  - Recover $f$
  - Embedding messages
    - $f = p dot f' + 1$
    - $c = g/f + q/p dot m$
    - $c dot f equiv g + q/p dot m dot p dot f' + q/p dot m$
  - Alternative:
    - $h = p dot g / f$, $c = r dot h + m$
  - Parameters matter: NTRU fatigue/overstretched NTRU
]
#slide(title: [NTRU lattice])[
  $
    mat(
      1, h;
      0, q;
    )
  $
  - Matrix form for $h$ (and $0$, $1$, $q$)
    - (anti-)circulant matrix
  - short vector: $(f, g) = f dot (1, h) + k dot (0, q)$
]

#slide(title: [Estimating difficulty])[
  - https://github.com/malb/lattice-estimator
    - Viability
    - Ideas for attacks to look at
    - Making sure your own lattices are safe?
  - https://github.com/WvanWoerden/NTRUFatigue
    - Specifically for NTRU
    - Fatigue/overstretched regime
]

#slide(title: [Breaking things])[
  === Linear algebra
  - Basis is linear algebra + noise
  - Sometimes the noise is not there
  - Or not enough
  - So just throw matrices at it
  === Lattice reduction
  - AKA primal attack
  - The straightforward thing
]

#slide(title: [Breaking things])[
  === Weak structure
  - RLWE/NTRU/... in a weird ring
    - Composite modulus
    - Reducible polynomial
    - ...
  - Chinese remainder theorem
    - AKA working mod a factor
  - Depends on end goal
]

#slide(title: [Breaking things])[
  === Linearization
  - Arora-Ge attack
  - Consider $e in {-1, 0, 1}$
  - Write $v = lr(angle.l vec(s), va angle.r) - b$
  - $v dot (v - 1) dot (v + 1) equiv 0$
  - Max degree: 3
  - Enough samples $->$ each monomial becomes 1 #alert[linear] variable
  - Linear algebra
]

#focus-slide[
  = Some resources
  #h(2em)
  #set align(left)
  #set text(size: 16pt)
  - https://eprint.iacr.org/2023/032.pdf
  - https://eprint.iacr.org/2020/1506.pdf
  - https://kel.bz/post/lll/
  - https://github.com/rkm0959/Inequality_Solving_with_CVP
  - https://github.com/jvdsn/crypto-attacks
  - https://github.com/kionactf/coppersmith
  - https://gist.github.com/RobinJadoul/796857fa33b118c17a4e54ff1b7ccfbe
  - https://doi.org/10.1007/3-540-44670-2_12
]

  
#new-section-slide[Get your hands dirty]
#slide(title: [mapsack])[
  == ImaginaryCTF (round 25)
_A multiplicative knapsack, kinda._

  / Author: Robin_Jadoul
  / Flag format: `ictf{...}`
]
#slide(title: [tan])[
  == ImaginaryCTF 2023
_`tan(x)` is a broken hash function in terms of collision resistance and second preimage resistance. But you surely can't find the preimage of `tan(flag)`, right?_

  / Author: maple3142
  / Flag format: `ictf{...}`

]
#slide(title: [flagtor])[
  == ImaginaryCTF 2023
_I threw in a bit of source-given rev, because why not._

_> I hate crypto and rev because both are math_

_Sorry to people who feel like this and even say so in the ictf discord ;)_

  / Author: Robin_Jadoul
  / Flag format: `ictf{...}`
]
#slide(title: [Tough decisions])[
  == ECSC 2023
  _Champagne for my real friends, real pain for my sham friends._

  / Author: Robin_Jadoul
  / Flag format: `ECSC{...}`
]
#slide(title: [Unbalanced])[
  == ICC 2022
  _I want to keep my private key small, but I've heard this is dangerous. I think I've found a way around this though!_

  / Author: jack
  / Flag format: `ICC{...}`
]
#slide(title: [LeaK])[
  == pbctf 2020
  _I know there's a famous attack on biased nonces. Then, how about this?_

  / Author: rbtree
  / Flag format: `pbctf{...}`
  / Extra note: (try the harder approach, just for fun)
]
#slide(title: [Seed Me])[
  == pbctf 2021
  _I came up with this fun game that only lucky people can win. Do you feel lucky?_

  / Author: UnblvR
  / Flag format: `pbctf{...}`
]
#slide(title: [not new PRNG])[
  == SECCON finals 2022
  _Recently, I learned that this random number generator is called "MRG"._
  
  / Author: Xornet
  / Flag format: `SECCON{...}`
]
#slide(title: [onelinecrypto])[
  == SEETF 2023
  _How to bypass this line?_

  #show raw: set text(hyphenate: true)
  ```py
  assert __import__('re').fullmatch(r'SEE{\w{23}}', flag:=input()) and not int.from_bytes(flag.encode(), 'big') % 13**37
  ```

  / Author: Neobeo
  / Flag format: `SEE{...}`
]
#slide(title: [RNG+++])[
  == TSJ CTF 2022
  _I encrypted the flag and messages by xoring them with a random number generator again. But it should be harder to break this time._

  / Author: maple3142
  / Flag format: `TSJ{...}`
]
#slide(title: [Random Shuffling Algorithm])[
  == HITCON CTF 2023
  _I think you already know what is this challenge about after seeing the challenge name :)_

  / Author: maple3142
  / Flag format: `hitcon{...}`
]
#slide(title: [Reality (remake)])[
  == \<Mostly new\>
  _Based upon the challenge `reality` from google ctf 2019_

  / Author: Robin_Jadoul
  / Flag format: `flag{...}`
]
